<?php
/**
 *
 * SugarCRM Community Edition is a customer relationship management program developed by
 * SugarCRM, Inc. Copyright (C) 2004-2013 SugarCRM Inc.
 *
 * SuiteCRM is an extension to SugarCRM Community Edition developed by SalesAgility Ltd.
 * Copyright (C) 2011 - 2021 SalesAgility Ltd.
 *
 * MintHCM is a Human Capital Management software based on SuiteCRM developed by MintHCM, 
 * Copyright (C) 2018-2023 MintHCM
 *
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SUGARCRM, SUGARCRM DISCLAIMS THE WARRANTY
 * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License along with
 * this program; if not, see http://www.gnu.org/licenses or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 *
 * You can contact SugarCRM, Inc. headquarters at 10050 North Wolfe Road,
 * SW2-130, Cupertino, CA 95014, USA. or at email address contact@sugarcrm.com.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
 * these Appropriate Legal Notices must retain the display of the "Powered by SugarCRM" 
 * logo and "Supercharged by SuiteCRM" logo and "Reinvented by MintHCM" logo. 
 * If the display of the logos is not reasonably feasible for technical reasons, the 
 * Appropriate Legal Notices must display the words "Powered by SugarCRM" and 
 * "Supercharged by SuiteCRM" and "Reinvented by MintHCM".
 */

namespace SuiteCRM\Utility\AntiMalware\Providers;

use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerInterface;
use SuiteCRM\Exception\Exception;
use SuiteCRM\Exception\InvalidArgumentException;
use SuiteCRM\Exception\MalwareFoundException;
use SuiteCRM\Interfaces\AntiMalwareFileScanner;
use SuiteCRM\Utility\AntiMalware\AntiMalwareTrait;
use SuiteCRM\Utility\Configuration;
use SuiteCRM\Utility\SuiteLogger;

/**
 * Class Sophos
 * @package SuiteCRM\Utility\AntiMalware\Providers
 */
class Sophos implements AntiMalwareFileScanner, LoggerAwareInterface
{
    use AntiMalwareTrait;

    /**
     * @var bool $enabled
     */
    private $enabled = false;

    /**
     * @var string $path
     */
    private $path = '/opt/sophos-av/bin/savscan';

    /**
     * @var string $options
     */
    private $options = '-ss';

    /**
     * @var  LoggerInterface
     */
    private $logger;

    public function __construct()
    {
        $globalConfig = new Configuration();
        $configuration =
            $globalConfig['anti_malware_scanners'][self::class] ?? null;

        if (!empty($configuration)) {
            $this->enabled = $configuration['enabled'] ?? $this->enabled;
            $this->path = $configuration['path'] ?? $this->path;
            $this->options = $configuration['options'] ?? $this->options;
        }

        $this->logger = new SuiteLogger();
    }

    /**
     * Is the Anti-Malware FileScanner enabled and configured
     * @return bool true === yes, false === no
     */
    public function isAntiMalwareScannerAvailable(): bool
    {
        if ($this->isEnabled()) {
            return file_exists($this->getExecutablePath());
        }

        return false;
    }

    /**
     * @param string $path - location in the file system
     * @throws InvalidArgumentException
     * @throws MalwareFoundException|Exception
     */
    public function scanFilePath(string $path): void
    {
        $this->validatePath($path);

        $command = $this->getExecutablePath() . ' ' . $this->getOptions() . ' ' . escapeshellarg($path);
        $lineOutput = shell_exec($command);

        $this->processResponse($path, $lineOutput);
    }

    /**
     * @param string $path - location in the file system
     * @param $lineOutput
     * @throws Exception
     * @throws MalwareFoundException
     */
    private function processResponse(string $path, $lineOutput): void
    {
        $warningsFound = stripos($lineOutput, 'WARNING') !== false;
        if ($warningsFound) {
            $this->logger->warning($lineOutput);
            throw new Exception(
                '[SophosAntiVirus][Warnings Found]'
            );
        }

        $errorsFound = stripos($lineOutput, 'ERROR') !== false;
        if ($errorsFound) {
            $this->logger->error($lineOutput);
            throw new Exception(
                '[SophosAntiVirus][Errors Found]'
            );
        }

        $virusFound = stripos($lineOutput, 'FOUND') !== false;
        if ($virusFound) {
            $this->logger->emergency($lineOutput);
            throw new MalwareFoundException(
                '[SophosAntiVirus][MalwareDetected]' . $path
            );
        }
    }

    /**
     * @throws InvalidArgumentException
     */
    private function validatePath($path): void
    {
        if (!is_string($path)) {
            throw new InvalidArgumentException(
                '[SophosAntiVirus][scanPathForMalware] $path should be a string'
            );

        }

        if (!file_exists($this->getExecutablePath())) {
            throw new InvalidArgumentException(
                '[SophosAntiVirus][scanPathForMalware] executable path not found'
            );
        }

        if (!file_exists($path)) {
            throw new InvalidArgumentException(
                '[SophosAntiVirus][scanPathForMalware] ' . $path . ' not found'
            );
        }
    }

    /**
     * @param LoggerInterface $logger
     */
    public function setLogger(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    /**
     * @return boolean
     */
    private function isEnabled(): bool
    {
        return $this->enabled;
    }

    /**
     * @return string
     */
    private function getExecutablePath(): string
    {
        return $this->path;
    }

    /**
     * @return string
     */
    private function getOptions(): string
    {
        return $this->options;
    }
}
